# Livre : La régression logistique en épidémiologie
# ----------------------------------------------------------------------------------------------------
# # Programme de calcul des OR par classes après une transformation par fonctions splines (fonction principalement utilisée dans le chapitre 4 du livre)
# NB : il existe une autre fonction après une transformation par polynômes fractionnaires (ORcl_pf)
# ----------------------------------------------------------------------------------------------------
#
# Ce programme est appelé par la fonction ORcl.sp après modélisation de la variable X par des fonctions splines cubiques restreints réalisée avec rcspline.eval() du package Hmisc, suivie d'un modèle de régression logistique avec la variable X transformée et éventuellement d'autres variables.
#
# Les arguments de la fonction sont :
# ORcl_sp(x,spl,reglog,ref,cl)
#
# - x : nom de la variable transformée en fonctions splines cubiques restreints (à mettre entre "")
# - spl : résultat de la transformation en fonctions splines par rcspline.eval()
# - reglog : résultat du modèle de régression logistique
# - ref : référence (en général, centre de la classe de référence)
# - cl : valeurs pour lesquelles on veut les OR par rapport à ref (en général, centres des classes pour lesquelles on veut les OR). Doit être mis sous la forme cl=c(x1,x2,...)
#
# Exemple avec le fichier cycles3 utilisé dans le chapitre 4
# rcs <- rcspline.eval(cycles3$age, nk = 3, inclx = TRUE)  # construction des splines
# mrcs <- glm(acc ~ rcs+ovo, family = binomial(), data = cycles3)  # Modèle logistique
# ORclsp(x="age",spl=rcs,reglog=mrcs,ref=27,cl=c(17,22))
#  ----------------------------------------------------------------------------------------------------
# 
 ORcl_sp <- function(x,spl,reglog,ref,cl)  {

   library(epiDisplay)
   library(biostat3)
   library(car)
   library(Hmisc)
   library(splines)
   
   
noeuds <- attr(spl, "knots")  # Valeurs des noeuds pour les splines
nk <- length(noeuds)  # Nombre de noeuds
nsp <- nk - 1  # Nombre de fonctions splines
nomvar<-x

ref <- ref
cl <- cl
if (ref %in% cl) {
  stop("ref ne doit pas faire partie de cl")
} 

# Un des problèmes est que les splines créées par rcspline.eval() n'ont pas de noms, sauf la première qui est X. En conséquence les variables du modèle logistique mrcs s'appellent toutes pareil (ici rcs3). Il faut donc changer les noms de ces variables pour les distinguer.

# 1. Convertir rcs en data frame pour accéder aux colonnes
rcs_df <- as.data.frame(rcs)

# 2. Renommer les colonnes dans rcs_df pour les appeler rcs_1, rcs_2, ...
colnames(rcs_df) <- c("x", paste0("rcs_", 1:(ncol(rcs_df) - 1)))

# 2.1 Construction de la partie de la formule de glm qui doit remplacer rcs
formule.spl<-"x"  # variables splines
for (i in 2:length(colnames(rcs_df))) {
  formule.spl <- paste0(formule.spl, " + ", colnames(rcs_df)[i])
}

# 3. Calcul du modèle logistique en remplaçant rcs par les noms plines renommées dans rcs_df. Il faut remplacer la formule y ~ x + rcs + ... par y ~ x + rcs_1 + rcs_2 + ... + ...
# c'est sur ce modèle logistique que sera aplliquée lincom

# 3.1 formule initiale de glm (avec rcs) transformée en caractères et décomposition en tokens pour remplacer rcs
formule.mrcs.c <- as.character(reglog[["formula"]])  
formule.mrcs.tok <- strsplit(formule.mrcs.c, "rcs")

# 3.2 Construction de la formule pour glm (formule.glm) en remplaçant rcs par rcs_1, rcs_2, ...
if (is.na(formule.mrcs.tok[[3]][2])) {  # cas où il n'y a d'autres variables que X transformée en splines
  formule.glm<-paste(formule.mrcs.tok[[2]],formule.mrcs.tok[[1]],formule.spl)
} else {
  formule.glm<-paste(formule.mrcs.tok[[2]],formule.mrcs.tok[[1]],formule.spl,formule.mrcs.tok[[3]][2])
}

# 3.3 Annonce des résultats (mise ici pour ne pas être répétée à cause des boucles ultérieures)
cat("Modèle logistique où",nomvar,"est remplacé par",nsp,"fonctions splines (",nk,"noeuds)\n",
    "OR et IC pour la variable age ajustée sur les autres variables du modèle\n\n")

# 3.4 Calcul du modèle logistique
mrcs_b <- glm(formule.glm , 
              family = binomial(), data = cbind(cycles3, rcs_df))

# 4. Calcul des fonctions splines au centre de chaque classe pour utiliser dans lincom

# 4.1 Boucle pour chaque classe
for (i in cl) {  # Valeurs des fonctions splines pour la référence et pour la classe (i)
  Xpred.27 <- rcspline.eval(c(ref, i), knots = noeuds, inclx = TRUE)
  dfX <- numeric(nsp)  # Initialisation d'un vecteur des différences entre les splines à ref et à (i)
  for (j in 1:nsp) {  # Boucle sur les différentes fonctions splines
    dfX[j] <- Xpred.27[2, j] - Xpred.27[1, j]
  }

  # Noms des variables du modèle glm pour servir dans lincom
  # On sait que var_name[1] est "(Intercept)" suivi de X et des noms des variables splines
  var_name <- names(coef(mrcs_b))
  
  # Si les noms des variables sont génériques, on peut indexer directement dans var_name
  
  formule_lincom <- paste0(dfX[1], "*x ")
  for (j in 2:nsp) {  # boucle sur les fonctions splines (= nb noeuds - 1)
    formule_lincom <- paste0(formule_lincom, " + ", paste0(dfX[j], "*",var_name[j+1]))
  }
  formule_lincom <- paste0(formule_lincom, " = 0")
  
  # Affichage de la formule pour lincom (facultatif, pour vérification)
  #print(formule_lincom)
  
  # Appliquer lincom avec la formule
  combi.lin <- lincom(mrcs_b, formule_lincom, eform = TRUE)
  
  # Afficher les résultats
  cat("ref = ", ref, " classe = ", i, 
      " OR = ", sprintf("%.2f", combi.lin$Estimate),
      " 95% IC : [", sprintf("%.2f", combi.lin$`2.5 %`), 
      " - ", sprintf("%.2f", combi.lin$`97.5 %`), "]\n")
} # Fin de la boucle pour chaque classe


} # fin du programme


